//
//                      COPYRIGHT NOTICE
//
//      This code was originally written, and is owned by Brian Somers.
//      It may be copied, altered, given away or sold by anybody who
//      feels so inclined, but must at all times contain this copyright
//      notice.
//      brian@awfulhak.demon.co.uk
//
//
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "defaults.inc"
#	endif
#   endif
#else
#   include "defaults.h"
#   include "common/string.h"
#   include "common/absolute.h"
#   include "include/malloc.h"
#   include "include/stdio.h"
#   include "include/memory.h"
#endif

#if defined(__ZTC__) || __GNUC__ > 1
#   define OBRACE
#   define CBRACE
#else
#   define OBRACE {
#   define CBRACE }
#endif

String default_directive[NDEFAULTS] = { 	// prefix is always '#'
    OBRACE "***" CBRACE,
    OBRACE "code_name" CBRACE,
    OBRACE "code_needs" CBRACE,
    OBRACE "compile" CBRACE,
    OBRACE "derive" CBRACE,
    OBRACE "include_line" CBRACE,
    OBRACE "include_line_cont" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "interface_name" CBRACE,
    OBRACE "interface_needs" CBRACE,
    OBRACE "load" CBRACE,
    OBRACE "lock_name" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "object_name" CBRACE,
    OBRACE "object_needs" CBRACE,
    OBRACE "response_line" CBRACE,
    OBRACE "response_line_cont" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "target" CBRACE
};

String default_value[NDEFAULTS] = {
    OBRACE "$*.exe" CBRACE,
    OBRACE "$*.c" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "$(CC) -c $#$<" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "#include \"$#$@\"" CBRACE,
    OBRACE "" CBRACE,
    OBRACE "$*.inc" CBRACE,
    OBRACE "$*.h" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "$(CC) -e$@ @$&" CBRACE,
    OBRACE "$*.lck" CBRACE,
    OBRACE "$*.mk" CBRACE,
    OBRACE "$*.obj" CBRACE,
    OBRACE "***" CBRACE,
    OBRACE "$#$@" CBRACE,
    OBRACE "" CBRACE,
    OBRACE "$*.rsp" CBRACE,
    OBRACE "***" CBRACE
};

default_t::default_t(){
    this->directive_overrides = 0;
    this->value_overrides = 0;
    int f;
    for(f = 0; f < NDEFAULTS; f++)
	baltree_construct(&this->defines[f],0);
}

default_t::~default_t(){
    if(this->directive_overrides) mem_free(this->directive_overrides);
    if(this->value_overrides) mem_free(this->value_overrides);
    int f;
    for(f = 0; f < NDEFAULTS; f++)
	baltree_destroy(&this->defines[f]);
}

const String &default_t::directive(default_type type){
    if(!this->directive_overrides || !this->directive_overrides[(int)type])
	return default_directive[(int)type];
    return *this->directive_overrides[(int)type];
}

const String &default_t::value(default_type type){
    if(!this->value_overrides || !this->value_overrides[(int)type])
	return default_value[(int)type];
    return *this->value_overrides[(int)type];
}

void default_t::set_directive(default_type type,const String &value){
    if(!this->directive_overrides)
	this->directive_overrides =
	    (String **)mem_calloc(NDEFAULTS * sizeof(String *));
    this->directive_overrides[(int)type] = &this->allocate.stringref(value);
}

void default_t::set_value(default_type type,String &value){
    if(!this->value_overrides)
	this->value_overrides =
	    (String **)mem_calloc(NDEFAULTS * sizeof(String *));
    this->do_defines(value,type);
    this->value_overrides[(int)type] = &this->allocate.stringref(value);
}

static int isline(String &line,const String &want){
    const char *l = line;
    size_t offset;
    for(offset = 1; offset < line.len(); offset++)
	if(l[offset] != ' ' && l[offset] != '\t') break;
    if(offset < line.len() && !String::subcmp(line,offset,want,0,want.len()))
	if(line.len() - offset == want.len()){
	    line.setlen(0);
	    return 1;
	}else switch(line[offset += want.len()]){
	    case ' ':
	    case '\t':
		while(++offset < line.len())
		    switch(line[offset]){
			case ' ':
			case '\t':
			    break;
			default:
			    line.del(0,offset);
			    offset = line.len();
			    while(offset--)
				switch(line[offset]){
				    case ' ':
				    case '\t':
					break;
				    default:
					line.setlen(offset + 1);
					return 1;
				}
			    // Can't be reached
			    line = "default_t: scribble time !";
			    return 1;
		    }
		line.setlen(0);
		return 1;
	}

    return 0;
}

inline int labelchar(char ch){
    return (ch > 'a' && ch < 'z') || (ch > 'A' && ch < 'Z')
	|| (ch > '0' && ch < '9') || ch == '_';
}

void default_t::do_defines(String &line,default_type type){
    _balanced_tree *tree = &this->defines[(int)type];
    String *data;
    baltree_last(tree);	// This direction gets sub-strings afterwards
    while(data = (String *)baltree_prev(tree), data){
	const _balanced_tree_key *k = baltree_key(tree);
	int offset = 0;
	size_t ret;
	while(ret = line.chr(*(char *)k->data,offset), ret != BAD_SIZE_T){
	    offset += ret;
	    if(line.len() < offset + k->length) break;
	    if(!memcmp((char *)line + offset,(char *)k->data,k->length)
	       && (!offset || !labelchar(line[offset-1]))
	       && (line.len() == offset + k->length
		   || !labelchar(line[offset + k->length]))){
		line.del(offset,offset + k->length);
		line.insert(offset,*data);
		offset += k->length;
	    }else offset++;
	}
    }
}

default_type default_t::evaluate(String &line){
    if(line[0] != '#') return INVALID_TYPE;
    default_type ret = INVALID_TYPE;
    if(isline(line,this->directive(CODE_NEEDS)))
	ret = CODE_NEEDS;
    else if(isline(line,this->directive(INTERFACE_NEEDS)))
	ret = INTERFACE_NEEDS;
    else if(isline(line,this->directive(COMPILE)))
	ret = COMPILE;
    else if(isline(line,this->directive(DERIVE)))
	ret = DERIVE;
    else if(isline(line,this->directive(LOAD)))
	ret = LOAD;
    else if(isline(line,this->directive(LOCK_NAME)))
	ret = LOCK_NAME;
    else if(isline(line,this->directive(INCLUDE_LINE)))
	ret = INCLUDE_LINE;
    else if(isline(line,this->directive(INCLUDE_LINE_CONT)))
	ret = INCLUDE_LINE_CONT;
    else if(isline(line,this->directive(OBJECT_NAME)))
	ret = OBJECT_NAME;
    else if(isline(line,this->directive(INTERFACE_NAME)))
	ret = INTERFACE_NAME;
    else if(isline(line,this->directive(CODE_NAME)))
	ret = CODE_NAME;
    else if(isline(line,this->directive(RESPONSE_LINE)))
	ret = RESPONSE_LINE;
    else if(isline(line,this->directive(RESPONSE_LINE_CONT)))
	ret = RESPONSE_LINE_CONT;
    else if(isline(line,this->directive(OBJECT_NEEDS)))
	ret = OBJECT_NEEDS;
    else if(isline(line,this->directive(TARGET)))
	ret = TARGET;
    if(ret != INVALID_TYPE) this->do_defines(line,ret);
    return ret;
}

static char directives[][20] = {
    { "BINARY_NAME" },
    { "CODE_NAME" },
    { "CODE_NEEDS" },
    { "COMPILE" },
    { "DERIVE" },
    { "INCLUDE_LINE" },
    { "INCLUDE_LINE_CONT" },
    { "INCLUDE_NAME" },
    { "INTERFACE_NAME" },
    { "INTERFACE_NEEDS" },
    { "LOAD" },
    { "LOCK_NAME" },
    { "MK_NAME" },
    { "OBJECT_NAME" },
    { "OBJECT_NEEDS" },
    { "RESPONSE_LINE" },
    { "RESPONSE_LINE_CONT" },
    { "RESPONSE_NAME" },
    { "TARGET" }
};

default_type default_t::type(const String &s){
    const char *str = s;
    int ret, f;
    for(f = 0; f < NDEFAULTS; f++){
	ret = strcmp(str,directives[f]);
	if(!ret) return (default_type)f;
	else if(ret < 0) break;
    }
    return INVALID_TYPE;
}

void default_t::define(default_type t,const String &from,const String &to,
		       const String &file,int lineno){
    _balanced_tree *tree = &this->defines[(int)t];
    String *s = (String *)baltree_get(tree,(char *)from,from.len());
    if(s)
	if(*s != to)
		fprintf( stderr, "%s: %d: Warning: Re-definition of \"%s\"\n",
				(char *)file, lineno, (char *)from );
	else return;
    s = &this->allocate.stringref(to);
    String &key = this->allocate.stringref(from);
    baltree_put(tree,(char *)key,key.len(),s);
}

#if defined( OS2 ) | defined( MSDOS )
#	define SLASH '\\'
#else
#	define SLASH '/'
#endif

void ExpandName( String &Name, const char *fmt )
{
	size_t f = 0;
	String FullName( fmt );
	char *Sep = strrchr( Name, SLASH );
	while( f = FullName.chr( '$', f ), f != BAD_SIZE_T )
		if( FullName[ ++f ] == '*' )
		{
			FullName.del( f - 1, f + 1 );
			if( Sep )
			{
				FullName.insert( f - 1, Sep + 1 );
				f += strlen( Sep + 1 ) - 1;
			}
			else
			{
				FullName.insert( f - 1, Name );
				f += Name.len() - 1;
			}
		}
	if( Sep && fmt && !isabsolute( fmt ) )
		FullName.insert( 0, Name, 0, Sep - (char *)Name + 1 );
	Name = FullName;
}
