//
//                      COPYRIGHT NOTICE
//
//      This code was originally written, and is owned by Brian Somers.
//      It may be copied, altered, given away or sold by anybody who
//      feels so inclined, but must at all times contain this copyright
//      notice.
//      brian@awfulhak.demon.co.uk
//
//
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "mk.inc"
#	endif
#   endif
#else
#   include "parse.h"
#   include "common/progname.h"
#   include "common/expand.h"
#   include "version.h"
#   include "include/sys/stat.h"
#   include "include/stdlib.h"
#   include "include/ctype.h"
#   include "include/stdio.h"
#endif

#ifdef MSDOS
// Watcom does this on the load line
#   ifdef __ZTC__
unsigned _stack = 20000;
#   endif
#   ifdef __TURBOC__
unsigned _stklen = 20000;
#   endif
#else
int getfull(String &name);
#endif

int startup(char *p,Parse &filelist,int &flags);
void pmodules(Parse &m,const char *modname);
void pfiles(Parse &m,const char *modname,int type);
int mkhelp(void);
void justusage(void);
int usage(const char *message = 0);
char *prog;
int nprocs(void);

#define DEFAULT_FLAGS (MOD_LOAD|MOD_ALL_COMPILE)

int main(int argc,char **argv){
    String startupfile, mapfile, build_string, OrigArgv = *argv;
    prog = progname(argv[0]);
    if(argc == 1) return usage();

    int flags = DEFAULT_FLAGS, ret = 0, lflag = 0, iflag = 0, getout;
    int Fflag = 0, Mflag = 0, sflag = 0, parallel = -1, f;
    char *build_command = 0;

    if( expand( &argc, &argv ) != 0 )
    {
	perror( "expand" );
	return 1;
    }

    for(f = 1; f < argc; f++)
		if(*argv[f] == '-'){
			char *ptr = argv[f] + 1;
			if(*ptr == '-')
				if(!*++ptr){
					f++;
					break;
				}else{
					unexpand( &argc, &argv );
					return usage("Invalid option --");
				}
			
			for(getout = 0; !getout && *ptr; ptr++)
				switch(*ptr){
				case 'c':
					flags |= MOD_ONE_COMPILE;
					flags &= ~MOD_ALL_COMPILE;
					if(!lflag) flags &= ~MOD_LOAD;
					break;

				case 'f':
					flags |= MOD_FORCE;
					break;

				case 'h':
					unexpand( &argc, &argv );
					return mkhelp();

				case 'i':
					iflag = 1;
					getout = 1;
					if(*(ptr+1)) startupfile = ptr + 1;
					else if(f + 1 == argc){
						fputs( prog, stderr );
						unexpand( &argc, &argv );
						return usage(": No argument to -i option");
					}else startupfile = argv[++f];
					break;

				case 'k':
					flags |= MOD_KEEPTMP;
					break;

				case 'l':
					flags |= MOD_LOAD;
					flags &= ~MOD_ALL_COMPILE;
					lflag = 1;
					break;

				case 'm':
					getout = 1;
					if(*(ptr+1)) mapfile = ptr + 1;
					else if(f + 1 == argc){
						fputs( prog, stderr );
						unexpand( &argc, &argv );
						return usage(": No argument to -m option");
					}else mapfile = argv[++f];
					break;

				case 'p':
					getout = 1;
					if(*(ptr+1)) parallel = atoi(ptr + 1);
					else if(f + 1 == argc){
						fputs( prog, stderr );
						unexpand( &argc, &argv );
						return usage(": No argument to -p option");
					}else parallel = atoi(argv[++f]);
					if(parallel < 1) parallel = 1;
					break;

				case 'q':
					flags |= MOD_QUIET;
					break;

				case 'r':
					flags |= MOD_RECURSE;
					break;

				case 's':
					sflag = 1;
					break;

				case 'v':
					flags |= MOD_VERBOSE;
					break;

				case 'L':
					flags |= MOD_REVERSE_LOAD;
					break;

				case 'M':
					Mflag = 1;
					break;

				case 'O':
					Fflag |= TARGET_FILES;
					break;

				case 'R':
					flags |= MOD_REMOVE;
					break;

				case 'S':
					Fflag |= SOURCE_FILES;
					break;

				case 'T':
					Fflag |= TMP_FILES;
					break;

				default:
					fprintf( stderr, "Invalid option -%c\n", *ptr );
					unexpand( &argc, &argv );
					return usage();
				}
		}else break;

    if(Mflag | Fflag &&
       ((Mflag && Fflag) || sflag
		|| (flags & ~(MOD_RECURSE|MOD_VERBOSE)) != DEFAULT_FLAGS)){
		unexpand( &argc, &argv );
		return usage("-MOST flags may only be used with -rv flags");
	}

    if(f == argc){
		if(flags == (MOD_VERBOSE | DEFAULT_FLAGS) && !iflag){
			unexpand( &argc, &argv );
			return version(prog);
		}
		if(flags & MOD_VERBOSE){
			fputs( prog, stderr );
			fputs( ": Warning: -v should be used alone when "
				  "no modules are given\n", stderr );
			unexpand( &argc, &argv );
			return version(prog);
		}
		unexpand( &argc, &argv );
		return usage();
    }

    if(flags & MOD_VERBOSE){
		version(prog);
		if(iflag)
			if(startupfile.len()){
				fprintf( stderr, "Reading startup file %s\n",
						(char *)startupfile );
			}else
				fputs( "Ignoring any startup file\n", stderr );
    }
    Parse filelist;
    if(iflag){
		if(startupfile.len())
			ret = filelist.startup(startupfile,flags);
    }else{
		startupfile = getenv("MK_RC");
		if(startupfile.len()) ret = filelist.startup(startupfile,flags);
		else ret = startup( OrigArgv, filelist, flags );
    }

	if( !ret ){
		unexpand( &argc, &argv );
		return 2;
	}
    ret = 0;

    if(sflag)
		if(flags & MOD_REMOVE)
#if defined( MSDOS ) | defined( OS2 )
			build_command = "del";
#else
			build_command = "rm";
#endif
		else{
			if(flags & MOD_FORCE) build_command = "mk -fc";
			else build_command = "mk -c";
			if(iflag){
				build_string = build_command;
				build_string += "i \"";
				build_string += startupfile;
				build_string += "\"";
				build_command = build_string;
			}
		}

    if(parallel == -1) parallel = nprocs();

    if(mapfile.len() && !filelist.set_map(mapfile,flags & MOD_VERBOSE)){
		unexpand( &argc, &argv );
		return 1;
	}

    for(; f < argc; f++){
#ifdef MSDOS
		char *Dot = strchr( argv[f], '.' );
		char *Slash1 = strchr( argv[f], '\\' );
		char *Slash2 = strchr( argv[f], '/' );
		if( Dot && Slash1 < Dot && Slash2 < Dot ){
			fprintf( stderr, "Assuming %s to mean module ", argv[f] );
			*Dot = '\0';
			fprintf( stderr, "%s\n", argv[f] );
		}
#endif
		switch(filelist.getlist(argv[f],flags)){
		case PARSE_OK:
			if(Mflag) pmodules(filelist,argv[f]);
			else if(Fflag) pfiles(filelist,argv[f],Fflag);
			else if(!filelist.compile(build_command,parallel)){
				fprintf( stderr, "%s: %s: failed compiling\n", prog, argv[f] );
				ret = 1;
			}
			break;

		case PARSE_FAIL:
			fprintf( stderr, "%s: %s: failed parsing module\n",
					prog, argv[f] );
			ret = 1;
			break;

		case PARSE_NOMODULE:
			fprintf( stderr, "%s: Don't know how to make %s\n",
					prog, argv[f] );
			ret = 1;
		}
    }
	unexpand( &argc, &argv );
    return ret;
}

int startup(char *p,Parse &filelist,int &flags){
    String startup(p);
#ifdef MSDOS
    char ch;
    startup.setlen(startup.len()-4);
    for(int f = startup.len(); f; f--)
		if(ch = startup[f-1], isupper(ch)) startup.set(f-1,tolower(ch));
		else if(ch == '/') startup.set(f-1,'\\');
#elif defined( OS2 )
	if( startup.chr( '\\' ) == BAD_SIZE_T && startup.chr('/') == BAD_SIZE_T )
		if( !getfull(startup) )
			return 0;
		else
		{
			char *Dot = strrchr( startup, '.' );
			if( Dot )
				startup.setlen( Dot - (char *)startup );
		}
#else
    	if(startup.chr('/') == BAD_SIZE_T && !getfull(startup))
			return 0;
#endif
    startup += ".rc";
    struct stat st;
    if(!stat(startup,&st)){
		if(flags & MOD_VERBOSE)
			fprintf( stderr, "Reading startup file %s\n", (char *)startup );
		return filelist.startup(startup,flags);
    }
    return 1;
}

#ifndef MSDOS
int getfull(String &name){
    char *envpath = getenv("PATH");
    if(!envpath) return 0;
    char *tok, *path = mem_new char[strlen(envpath)+1];
    String newname;
    struct stat st;
    strcpy(path,envpath);
#ifdef OS2
    if(tok = strtok(path,";"), tok)
#else
    if(tok = strtok(path,":"), tok)
#endif
	do{
		newname = tok;
#ifdef OS2
		newname += '\\';
		newname += name;
		if( !strchr( name, '.' ) )
			newname += ".exe";
#else
		newname += '/';
		newname += name;
#endif
		if(!stat(newname,&st) && st.st_mode&S_IFREG){
			name = newname;
			return 1;
		}
	}
#ifdef OS2
	while(tok = strtok((char *)0,";"), tok);
#else
	while(tok = strtok((char *)0,":"), tok);
#endif
    return 0;
}
#endif // MSDOS

void pmodules(Parse &m,const char *modname){
    if(modname){ fputs( modname, stderr ); fputs( ":\n", stderr ); }
    const char *name;
    while(name = m.nextmodule(), name)
		puts( name );
}

void pfiles(Parse &m,const char *modname,int type){
    if(modname){ fputs( modname, stderr ); fputs( ":\n", stderr ); }
    const char *name;
    while(name = m.nextfile(type), name)
		puts( name );
}

#if defined(sequent) || defined(_SEQUENT_)
extern "C" int cpus_online(void);
int nprocs(){
	return cpus_online() / 2;
}
#else
int nprocs(){
	return 1;
}
#endif

void justusage(){
    fputs( "Usage: ", stderr );
    fputs( prog, stderr );
    if( strcmp( prog, "mk" ) ) fputs( " [mk]", stderr );
    fputs( " [ -i ifile ] [ -m mfile ] [ -p n ] "
		  "[ -cfhklqrsvLMORST ] modules\n", stderr );
}

int mkhelp(){
    justusage();
    fputs( "       -c        - Compile only specified module.\n", stderr );
    fputs( "                   If recursing, compile library headers.\n",
		  stderr );
    fputs( "       -f        - Force all targets to be out of date.\n",
		  stderr );
    fputs( "       -h        - Display this message.\n", stderr );
    fputs( "       -i ifile  - Use \"ifile\" for initialization.\n", stderr );
    fputs( "       -k        - Keep temporary include and response files.\n",
		  stderr );
    fputs( "       -l        - Load only specified module.\n", stderr );
    fputs( "                   If recursing, load any target.\n", stderr );
    fputs( "       -m mfile  - Use \"mfile\" for mapping.\n", stderr );
    fputs( "       -p n      - Execute at most 'n' parallel compilations.\n",
		  stderr );
    fputs( "       -q        - Quiet mode - don't show executing commands.\n",
		  stderr );
    fputs( "       -r        - Recurse when encountering target directive.\n",
		  stderr );
    fputs( "       -s        - Show only, don't execute commands.\n", stderr );
    fputs( "       -v        - Verbose parse information is given.\n",
		  stderr );
    fputs( "                   If no module is given, just the version number"
		  " is shown.\n", stderr );
    fputs( "       -L        - Link modules in reverse order\n", stderr );
    fputs( "       -M        - Show module list.\n", stderr );
    fputs( "       -O        - Show object file list, including non existent"
		  " files.\n", stderr );
    fputs( "       -R        - Remove all out of date targets.\n", stderr );
    fputs( "       -S        - Show source file list.\n", stderr );
    fputs( "       -T        - Show temporary file list, including non"
		  " existent files.\n", stderr );
    fputs( "       modules - The list of modules to operate on.\n", stderr );
    return 0;
}

int usage(const char *message){
    if(message) fprintf( stderr, "%s: %s\n", prog, message );
    justusage();
    fputs( "       use ", stderr );
    fputs( prog, stderr );
    fputs( " -h for help\n", stderr );
    return -1;
}
