//                      COPYRIGHT NOTICE
//
//      This code was originally written, and is owned by Brian Somers.
//      It may be copied, altered, given away or sold by anybody who
//      feels so inclined, but must at all times contain this copyright
//      notice.
//      brian@awfulhak.demon.co.uk
//
//
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "parse.inc"
#	endif
#   endif
#else
#   include "parse.h"
#   include "common/absolute.h"
#   include "common/signals.h"
#   include "pr.h"
#   include "jobs.h"
#   include "readfile.h"
#   include "include/fcntl.h"
#   include "include/dos.h"
#   include "include/io.h"
#   include "include/prototyp.h"
#   include "include/dir.h"
#   include "include/stdlib.h"
#   include "include/ctype.h"
#   include "include/stdio.h"
#   include "include/errno.h"
#   include "include/memory.h"
#endif

#if defined( MSDOS ) | defined( OS2 )
#   define SLASH '\\'
static const char *Parent = "..\\";
#else
#   define SLASH '/'
static const char *Parent = "../";
#endif

#ifdef SEQUENTPTX
extern "C" FILE *fdopen( int, const char * );
#endif

class mkline {
    mkline *next;
public:
    String name;
    int lineno;
    int interface_needs;
	mkline *Next(void){ return next; }
    mkline(const String &s,int n,int l,mkline *m = 0);
	mkline *Add(const String &s,int n,int l);
	~mkline(void){ mem_delete this->next; }
};

mkline::mkline(const String &s,int n,int l,mkline *m) : name(s)
{
	interface_needs = n;
	lineno = l;
	next = m;
}

mkline *mkline::Add(const String &s,int n,int l){
	return next ? next->Add(s,n,l) : (next = mem_new mkline(s,n,l));
}

#define COMPILING_ONE (this->flags & MOD_ONE_COMPILE)
#define COMPILING_ALL (this->flags & MOD_ALL_COMPILE)
#define COMPILING (COMPILING_ONE | COMPILING_ALL)
#define RECURSING (this->flags & MOD_RECURSE)
#define LOADING (this->flags & MOD_LOAD)
#define FORCING (this->flags & MOD_FORCE)
#define VERBOSE (this->flags & MOD_VERBOSE)
#define KEEP (this->flags & MOD_KEEPTMP)
#define REMOVING (this->flags & MOD_REMOVE)
#define QUIET (this->flags & MOD_QUIET)
#define REVERSE_LOAD (this->flags & MOD_REVERSE_LOAD)

const String NOSTRING;

static const char *dogetenv(const String &s){
#if defined( MSDOS ) | defined( OS2 )
    String snew(s);
    int f = snew.len();
    const char *ss = snew;
    while(f--) if(islower(ss[f])) snew.set(f,toupper(ss[f]));
    return getenv(snew);
#else
    return getenv(s);
#endif
}

static void dogetcwd( String &dir ){
    dir.set(0,'\0');
    dir.setlen(502);
    getcwd(dir,500);
    dir.setlen(dir.chr(0));
#if defined( MSDOS ) | defined( OS2 )
    size_t f = dir.len();
    const char *cwd = dir;
    while(f--)
		if(isupper(cwd[f])) dir.set(f,tolower(cwd[f]));
		else if(cwd[f] == '/') dir.set(f,SLASH);
    if( dir[1] != ':' ){
		char drive[3];
		unsigned Disk;
		DOSgetdisk(&Disk);
		drive[0] = 'a' + (char)Disk;
		drive[1] = ':';
		drive[2] = '\0';
		dir.insert( 0, drive );
    }
#endif
}

// if dirlen is zero, the correct value will be returned (for parent)
// if dirlen is not supplied (-1), parent is taken as a directory name only
static int buildcompletename( String &name, const String &parent,
							 int dirlen = -1 ){
    if( !isabsolute( name ) )
#if defined( MSDOS ) | defined( OS2 )
	if( name[1] == ':' )
	{
	    fputs( "Warning: Relative paths with specific drives "
			  "are not supported.\n"
			  "         Possible parse errors may result.\n"
			  "         Name is specified as \"", stderr );
		fputs( name, stderr );
		fputs( "\"\n", stderr );
	    String otherdir;
	    unsigned current;
	    DOSgetdisk(&current);
	    DOSsetdisk( name[0] - 'a' );
	    dogetcwd( otherdir );
#if defined( MSDOS ) | defined( OS2 )
	    otherdir += "\\";
#else
	    otherdir += "/";
#endif
	    DOSsetdisk( current );
	    name.del( 0, 2 );
	    name.insert( 0, otherdir );
		fputs( "         Name changed to \"", stderr );
		fputs( name, stderr );
		fputc( '\n', stderr );
	}else
#endif
	{
	    switch( dirlen ){
		case 0:
		    dirlen = parent.rchr( SLASH );
		    break;
		case -1:
		    dirlen = parent.len();
	    }
	    name.insert( 0, parent, 0, dirlen );
#if defined( MSDOS ) | defined( OS2 )
	    name.insert( dirlen, "\\" );
#else
	    name.insert( dirlen, "/" );
#endif
	}

    name.realname();
#if defined( MSDOS ) | defined( OS2 )
    if( name[1] != ':' )
	name.insert( 0, parent, 0, 2 );
#endif
    return dirlen;
}

Parse::Parse(const char *modname,int f){
    baltree_construct(&this->modulexref,BALTREE_FAST_KEYS);
    baltree_construct(&this->env,BALTREE_MALLOC_KEYS);
    baltree_construct(&this->ignore,BALTREE_MALLOC_KEYS);
    this->map = 0;
    this->top_mod = 0;
    this->getlist(modname,f);
}

Parse::Parse(){
    baltree_construct(&this->modulexref,BALTREE_FAST_KEYS);
    baltree_construct(&this->env,BALTREE_MALLOC_KEYS);
    baltree_construct(&this->ignore,BALTREE_MALLOC_KEYS);
    this->map = 0;
    this->top_mod = 0;
}

Parse::~Parse(){
    if(this->map) mem_delete this->map;
    String *s;
    baltree_destroy(&this->ignore);
    baltree_first(&this->env);
    while(s = (String *)baltree_next(&this->env), s) mem_delete s;
    baltree_destroy(&this->env);
    baltree_destroy(&this->modulexref);
    int size = this->modulelist.size();
    while(size--) mem_delete (module *)this->modulelist.linkout();
}

void Parse::dellist(){
    if(!this->top_mod) return;
    this->top_mod = 0;
    String *s;
    baltree_first(&this->env);
    while(s = (String *)baltree_next(&this->env), s) mem_delete s;
    baltree_destroy(&this->env);
    baltree_destroy(&this->modulexref);
    int size = this->modulelist.size();
    while(size--) mem_delete (module *)this->modulelist.linkout();
    baltree_construct(&this->modulexref,BALTREE_FAST_KEYS);
    baltree_construct(&this->env,BALTREE_MALLOC_KEYS);
    this->cwd.setlen(0);
}

// Don't declare this inside ::Relative - compilers such as WATCOM never
// delete it (stupid compiler).  This is probably because it doesn't actually
// instantiate it before main() - this is WRONG too.
static String RelativeBuffer;

const String &Parse::Relative(const String &name){
    if(!name.ncmp(this->cwd,this->cwd.len())
	   && name[this->cwd.len()] == SLASH){
		RelativeBuffer = (const char *)name + this->cwd.len() + 1;
		return RelativeBuffer;
    }
    size_t pos = this->cwd.len(), n = 0;
    while(n++, pos = this->cwd.rchr(SLASH,pos-1), pos != BAD_SIZE_T)
		if(!name.ncmp(this->cwd,pos+1))
			if(n * 3 < pos){
				RelativeBuffer.setlen(0);
				while(n--) RelativeBuffer += Parent;
				RelativeBuffer += (const char *)name + pos + 1;
				return RelativeBuffer;
			}else break;
    RelativeBuffer = name;
    return RelativeBuffer;
}

void Parse::putenv(char *name,const char *value = ""){
    int len = strlen(name) + 1;
    String *s = (String *)baltree_get(&this->env,name,len);
    if(s) *s = value;
    else{
		s = mem_new String(value);
		baltree_insert(&this->env,name,len,s);
    }
}

const String &Parse::getenv(const String &name){
    String *s = (String *)
		baltree_get(&this->env,(const char *)name,name.len()+1);
    if(!s){
		s = mem_new String(dogetenv(name));
		baltree_insert(&this->env,(const char *)name,name.len()+1,s);
    }
    return *s;
}

void Parse::do_malloc(char *&c,String &l){
    this->dontexpandtarget();
    c = this->alloc.charptr(this->Expand(l));
}

void Parse::dontexpandtarget(void){
    this->putenv("*","$*");	// module name
    this->putenv("@","$@");	// target name
    this->putenv("<","$<");	// source name
    this->putenv("&","$&");	// include name
    this->putenv("#","$#");	// dir name
}

// Don't declare this inside ::SetTarget - compilers such as WATCOM never
// delete it (stupid compiler).  This is probably because it doesn't actually
// instantiate it before main() - this is WRONG too.
static String settargetBuffer(30), expandedBuffer(60);

void Parse::SetTarget(const String &mod, const String &dst,
					  const String &lst = NOSTRING,
					  const String &src = NOSTRING ){
    size_t off = mod.rchr(SLASH);
    if(off == BAD_SIZE_T) settargetBuffer = mod;
    else settargetBuffer = (const char *)mod + off + 1;
	expandedBuffer = settargetBuffer;
    this->putenv( "*", settargetBuffer);
    ExpandName( expandedBuffer, dst );
    this->putenv( "@", expandedBuffer );
    if(lst.len()){
		expandedBuffer = settargetBuffer;
		ExpandName( expandedBuffer, lst );
		this->putenv( "&", expandedBuffer );
    }else this->putenv( "&" );
    if( src.len() ){
		expandedBuffer = settargetBuffer;
		ExpandName( expandedBuffer, src );
		this->putenv( "<", expandedBuffer );
    }else this->putenv( "<" );
    if( off == BAD_SIZE_T ) settargetBuffer.setlen(0);
    else{
		settargetBuffer = mod;
		settargetBuffer.setlen( off + 1 );
    }
    this->putenv( "#", settargetBuffer );
}

module *Parse::ReadModule(String &modname){
    module *m = (module *)baltree_get(&this->modulexref,
									  (const char *)modname,modname.len()+1);
    static module *recursion_error;

    if(m){
		if(!m->isvalid()){
			String Name( this->Relative( m->name() ) );
			ExpandName( Name, this->defaults.value( MK_NAME ) );
			fprintf( stderr, "%s: Recursion error:\n", (char *)Name );
			recursion_error = m;
		}
		return m;				// Beginning of recursion error
    }

	int dirlen = 0, fd, toplev, needhalfmap = 0;
	module *new_m;
	char *binfmt = 0;

	toplev = this->top_mod ? 0 : 1;
	m = mem_new module(modname,this->defaults,toplev);
	if(!this->top_mod) this->top_mod = m;
	fd = m->openread( this->defaults.value( MK_NAME ), this->map, VERBOSE );
	if(this->map && toplev) m->reversemap(); // Just in case I recurse
	baltree_insert(&this->modulexref,m->realname(),m->reallen(),m);

	if(fd != -1){
		mkline *FirstLine = 0, *Mline = 0;
		readfile file(fd);		// Buffer the lot
		close(fd);				// for recursion (file descriptors)
		String line(80);
		int got_responseline = 0, got_responselinecont = 0;
		int lineno = 0;
		m->has_mk = 1;
		while(lineno++, file.getline(line) != EOF){
			if(line[0] != '#') continue;
			baltree_setafter(&this->ignore,line,line.len()+1);
			if(baltree_prev(&this->ignore)){
				const _balanced_tree_key *k = baltree_key(&this->ignore);
				if( line.len() >= k->length
				   && !memcmp( k->data, line, k->length ) )
					continue;
			}
			switch( this->defaults.evaluate( line ) ){
				case CODE_NEEDS:
				if(!this->Expand(line,1).len())
				{
					String Name( this->Relative( m->name() ) );
					ExpandName( Name, this->defaults.value( MK_NAME ) );
					fprintf( stderr, "%s:%d: Warning: "
							"null module name ignored\n",
							(const char *)Name, lineno );
				}
				else if( Mline )
					Mline = Mline->Add(line,0,lineno);
				else
					FirstLine = Mline = mem_new mkline(line,0,lineno);
				break;

				case INTERFACE_NEEDS:
				if(!this->Expand(line,1).len())
				{
					String Name( this->Relative( m->name() ) );
					ExpandName( Name, this->defaults.value( MK_NAME ) );
					fprintf( stderr, "%s:%d: Warning: "
							"null module name ignored\n",
							(const char *)Name,
							lineno );
				}
				else if( FirstLine )
					// Add first so that includes are in the correct order
					FirstLine = mem_new mkline(line,1,lineno,FirstLine);
				else
					FirstLine = Mline = mem_new mkline(line,1,lineno);
				break;

				case OBJECT_NEEDS:
				if(!this->Expand(line,1).len())
				{
					String Name( this->Relative( m->name() ) );
					ExpandName( Name, this->defaults.value( MK_NAME ) );
					fprintf( stderr, "%s:%d: Warning: "
							"null module name ignored\n",
							(const char *)Name, lineno );
				}
				else if(LOADING){
					dirlen = buildcompletename( line, modname, dirlen );
					new_m = (module *)baltree_get(&this->modulexref,
												  (const char *)line,
												  line.len()+1);
					if(!new_m){
						new_m = mem_new module(line,this->defaults);
						new_m->object_namefmt = "$*";
						new_m->orig_code_time = new_m->code_time = 0;
						new_m->orig_object_time = new_m->object_time = 1;
						baltree_insert(&this->modulexref,new_m->realname(),
									   new_m->reallen(),new_m);
					}
					m->direct_objects.list.linkbefore(new_m);
				}
				break;

				case DERIVE:
				if( m->derive )
				{
					String Name( this->Relative( m->name() ) );
					ExpandName( Name, this->defaults.value( MK_NAME ) );
					fprintf( stderr, "%s:%d: Warning: "
							"Additional derive line ignored\n",
							(const char *)Name, lineno );
				}
				else if( ( COMPILING_ONE && toplev ) || COMPILING_ALL )
				{
					size_t Space = line.chr( ' ' );
					size_t Tab = line.chr( '\t' );
					if( Tab != BAD_SIZE_T
						&& ( Space == BAD_SIZE_T || Tab < Space ) )
						Space = Tab;
					if( Space == BAD_SIZE_T )
					{
						String Name( this->Relative( m->name() ) );
						ExpandName( Name, this->defaults.value( MK_NAME ) );
						fprintf( stderr, "%s:%d: Warning: "
								"Invalid derivation format ignored\n",
								(const char *)Name, lineno );
					}
					else
					{
						// As 'line' has a space, but neither begins or ends
						// with a space, it must constitute at least two strings
						String Namefmt( line, Space++ );
						while( line[ Space ] == ' ' || line[ Space ] == '\t' )
							Space++;
						line.del( 0, Space );
						m->derive = mem_new derive_t;
						this->do_malloc( m->derive->namefmt, Namefmt );
						this->do_malloc( m->derive->compile, line );
					}
				}
				break;

				case COMPILE:
				this->do_malloc(m->compile,line);
				break;

				case LOAD:
				this->do_malloc(m->load,line);
				break;

				case INCLUDE_LINE:
				this->do_malloc(m->includeline,line);
				break;

				case INCLUDE_LINE_CONT:
				this->do_malloc(m->includelinecont,line);
				break;

				case OBJECT_NAME:
				this->do_malloc(m->object_namefmt,line);
				break;

				case INTERFACE_NAME:
				this->do_malloc(m->interface_namefmt,line);
				break;

				case CODE_NAME:
				this->do_malloc(m->code_namefmt,line);
				break;

				case RESPONSE_LINE:
				got_responseline = 1;
				this->do_malloc(m->responseline,line);
				break;

				case RESPONSE_LINE_CONT:
				got_responselinecont = 1;
				this->do_malloc(m->responselinecont,line);
				break;

				case TARGET:
				if(!m->target)
				{
					m->target = mem_new target_t;
					if(line.len()) this->do_malloc(m->target->namefmt,line);
				}
				else
				{
					String Name( this->Relative( m->name() ) );
					ExpandName( Name, this->defaults.value( MK_NAME ) );
					fprintf( stderr, "%s:%d: repeated target ignored\n",
							(const char *)Name, lineno );
				}
				break;

				default:
				String Name( this->Relative( m->name() ) );
				ExpandName( Name, this->defaults.value( MK_NAME ) );
				fprintf( stderr, "%s:%d: \"%s\": Invalid directive\n",
						(const char *)Name,
						lineno, (const char *)line );
			}
		}
		
		if( m->target )
		{
			if( !m->target->namefmt )
				binfmt = this->defaults.value( BINARY_NAME );
			if( !RECURSING && !toplev )
			{
				mem_delete FirstLine;
				if( m->target->namefmt )
				{
					// I'm a library and we're not recursing
					m->object_time = m->code_time =
						m->orig_object_time = m->orig_code_time = 0;
					if( m->gettime(H_F,this->map,VERBOSE) && !FORCING )
						m->interface_time = m->orig_interface_time;
					else m->interface_time = OUTOFDATE;
					m->newest_interface = m->interface_time;
					if( m->gettime(A_F,this->map,VERBOSE,binfmt) && !FORCING )
						m->target->a_time = m->target->orig_a_time;
					else m->target->a_time = OUTOFDATE;
					m->direct_objects.newest = m->target->a_time;
				}
				else
				{
					// I'm not a library and we're not recursing - bad news !
					m->newest_interface = m->interface_time = m->code_time =
						m->object_time = m->orig_interface_time =
							m->orig_code_time = m->orig_object_time = 0;
					fprintf( stderr, "Warning: target executable "
							"module \"%s\" ignored by non-recursive parse\n", 
							(const char *)this->Relative(m->name()) );
				}
				this->modulelist.linkbefore(m);
				m->makevalid();
				return m;				// !RECURSING && !toplev && target
			}
		}else{
			if(got_responseline)
			{
				String Name( this->Relative( m->name() ) );
				ExpandName( Name, this->defaults.value( MK_NAME ) );
				fprintf( stderr, "Warning: Useless \"%s\" directive "
						"found in %s\n",
						(const char *)this->defaults.directive(RESPONSE_LINE),
						(const char *)Name );
			}
			if(got_responselinecont)
			{
				String Name( this->Relative( m->name() ) );
				ExpandName( Name, this->defaults.value( MK_NAME ) );
				fprintf( stderr, "Warning: Useless \"%s\" directive "
						"found in %s\n", (const char *)
						this->defaults.directive(RESPONSE_LINE_CONT),
						(const char *)Name );
			}
		}

		_balanced_tree sorted;
		baltree_construct(&sorted,BALTREE_MALLOC_KEYS|BALTREE_FAST_KEYS);
		if(VERBOSE && !FirstLine)
		{
			String Name( this->Relative( m->name() ) );
			ExpandName( Name, this->defaults.value( MK_NAME ) );
			fprintf( stderr, "%s (%d lines) has no dependencies\n",
					(const char *)Name, lineno-1 );
		}
		for(Mline = FirstLine; Mline; Mline = Mline->Next()){
			dirlen = buildcompletename( Mline->name, modname, dirlen );
			if(VERBOSE){
				String Name( this->Relative( m->name() ) );
				ExpandName( Name, this->defaults.value( MK_NAME ) );
				fputs( Name, stderr );
				fputc( ' ', stderr );
				if(Mline->interface_needs)
					fputs( this->defaults.directive(INTERFACE_NEEDS), stderr );
				else
					fputs( this->defaults.directive(CODE_NEEDS), stderr );
				fputc( ' ', stderr );
				fputs( this->Relative(Mline->name), stderr );
				fputc( '\n', stderr );
			}
			if(new_m = this->ReadModule(Mline->name), !new_m){
				String Name( this->Relative( m->name() ) );
				ExpandName( Name, this->defaults.value( MK_NAME ) );
				fputs( Name, stderr );
				fprintf( stderr, ": %d: Warning: %s: no such module"
						" - ignored\n", Mline->lineno,
						(const char *)this->Relative(Mline->name) );
				continue;
			}
			
			if(!new_m->isvalid()){
				if(recursion_error){
					String Name( this->Relative( m->name() ) );
					ExpandName( Name, this->defaults.value( MK_NAME ) );
					fputs( Name, stderr );
					fputc( ' ', stderr );
					if( Mline->interface_needs )
						fputs( this->defaults.directive(INTERFACE_NEEDS),
							  stderr );
					else
						fputs( this->defaults.directive(CODE_NEEDS), stderr );
					fputc( ' ', stderr );
					fputs( this->Relative(new_m->name()), stderr );
					fputc( '\n', stderr );
					if(m == recursion_error) recursion_error = 0;
				}
				mem_delete FirstLine;
				baltree_destroy(&sorted);
				return m;				// Middle of recursion error
			}
			if( new_m->hasmapped() ){
				m->now_hasmapped();
				if( new_m->hasinterfacemapped() ){
					needhalfmap = 1;
					if( Mline->interface_needs )
						m->now_hasinterfacemapped();
				}
			}
			if(LOADING && (!new_m->target || new_m->target->namefmt)){
				m->direct_objects.list.linkbefore(new_m);
				if(m->direct_objects.newest < new_m->direct_objects.newest)
					m->direct_objects.newest = new_m->direct_objects.newest;
				if(new_m->target && m->direct_objects.newest
				   < new_m->target->a_time )
					m->direct_objects.newest = new_m->target->a_time;
			}
			if(!COMPILING && (!m->target || !m->target->namefmt || !LOADING))
				// Don't want any headers
				continue;
			if(COMPILING_ONE && !toplev && !Mline->interface_needs
			   && (!RECURSING || !m->target))
				// Don't want used headers.  NOTE: can't mem_delete new_m lists
				// because they may be #needed via another route
				continue;
			hlist &h = Mline->interface_needs ? m->interface_needs :
				m->code_needs;
			hlist &sub_needs = new_m->interface_needs;
			if(!new_m->target){
				if(h.newest < sub_needs.newest) h.newest = sub_needs.newest;
				if(h.newest < new_m->interface_time)
					h.newest = new_m->interface_time;
			}else if(h.newest < new_m->newest_interface)
				h.newest = new_m->newest_interface;
			if( !new_m->target ){
				int sz = sub_needs.list.size();
				module *add;
				for(sub_needs.list.start(); sz--; sub_needs.list.fwd()){
					add = (module *)sub_needs.list.get();
					// YES, we're adding the pointer itself to the list
					if(baltree_insert(&sorted,&add,sizeof(add),add))
						h.list.linkafter(add);
				}
			}
			if(baltree_insert(&sorted,&new_m,sizeof(new_m),new_m)
			   && new_m->interface_time)
				h.list.linkafter(new_m);
		}
		mem_delete FirstLine;
		baltree_destroy(&sorted);
    }

	if( !m->ismapped() )
		if( needhalfmap )
			m->HalfMap(this->map,VERBOSE,1);
		else if( m->target && m->hasmapped() )
			m->HalfMap(this->map,VERBOSE);

	if( m->derive )
	{
		m->gettime(D_F,this->map,VERBOSE);
		if( !m->derive->src_time )
		{
			String Name( m->name() );
			ExpandName( Name, m->derive->namefmt );
			fprintf( stderr, "Cannot locate %s !\n", (const char *)Name );
			baltree_del(&this->modulexref,m->realname(),m->reallen());
			mem_delete m;
			if(toplev) this->top_mod = 0;
			return 0;				// Non existent module
		}
		if( m->target && m->target->namefmt )
		{
			mem_delete m->derive;
			m->derive = 0;
			if( VERBOSE )
				fputs( "Ignore derive directive for library target\n", stderr );
		}
	}

	m->gettime( H_F, this->map, VERBOSE );
	if( ( m->target && m->target->namefmt
			&& ( !m->orig_interface_time || FORCING ) )
		|| ( m->derive
			&& ( m->orig_interface_time < m->derive->src_time || FORCING ) ) )
	{
		if(VERBOSE){
			String Name( m->name() );
			ExpandName( Name, m->interface_namefmt );
			fputs( Name, stderr );
			fputs( ": ", stderr );
			if(!m->interface_time) fputs( "Non existent\n", stderr );
			else fputs( "Out of date\n", stderr );
		}
		m->interface_time = OUTOFDATE;
    }
	else
		m->interface_time = m->orig_interface_time;

	m->newest_interface = m->interface_time;
	if(m->newest_interface < m->interface_needs.newest)
		m->newest_interface = m->interface_needs.newest;
	if(m->newest_interface < m->code_needs.newest)
		m->newest_interface = m->code_needs.newest;
	if( m->target && m->target->namefmt )
	{
		if(m->newest_interface != m->interface_time)
		{
			if(VERBOSE)
			{
				String Name( m->name() );
				ExpandName( Name, m->interface_namefmt );
				fprintf( stderr, "%s: Out of date\n", (char *)Name );
			}
			m->newest_interface = m->interface_time = OUTOFDATE;
		}
		// code/object/derive not allowed for a library
		m->orig_object_time = m->object_time = m->code_time = 0;
    }
	else if( m->gettime( C_F, this->map, VERBOSE ) || m->derive )
	{
		// We (should) have a code file
		if( m->derive && ( FORCING || m->orig_code_time < m->derive->src_time 
						  || m->interface_time == OUTOFDATE ) )
		{
			if( VERBOSE )
			{
				String Name( m->name() );
				ExpandName( Name, m->code_namefmt );
				fputs( Name, stderr );
				if(!m->orig_code_time)
					fputs( ": Non existent\n", stderr );
				else
					fputs( ": Out of date\n", stderr );
			}
			m->interface_time = m->code_time = OUTOFDATE;
		}
		else
			m->code_time = m->orig_code_time;

		if( !m->gettime( O_F, this->map, VERBOSE )
		   || m->code_time == OUTOFDATE
		   || FORCING
		   || m->orig_object_time < m->newest_interface
		   || m->orig_object_time < m->orig_code_time )
		{
			if(VERBOSE)
			{
				String Name( m->name() );
				ExpandName( Name, m->object_namefmt );
				fputs( Name, stderr );
				if(!m->orig_object_time)
					fputs( ": Non existent\n", stderr );
				else
					fputs( ": Out of date\n", stderr );
			}
			m->object_time = OUTOFDATE;
		}
		else
			m->object_time = m->orig_object_time;
    }
	else
		// No code file, no object file
		m->code_time = m->orig_object_time = m->object_time = 0;
	
	if(m->direct_objects.newest < m->object_time)
		m->direct_objects.newest = m->object_time;
	
	if(m->target)
		if( !m->gettime( A_F, this->map, VERBOSE, binfmt ) || FORCING
		   || m->target->orig_a_time < m->direct_objects.newest )
		{
			if( VERBOSE )
			{
				String Tgt( m->name() );
				if( m->target->namefmt )
					ExpandName( Tgt, m->target->namefmt );
				else
					ExpandName( Tgt, this->defaults.value( BINARY_NAME ) );
				fputs( Tgt, stderr );
				if(m->target->orig_a_time) fputs( ": Out of date\n", stderr );
				else fputs( ": Non existent\n", stderr );
			}
			m->target->a_time = OUTOFDATE;
		}
		else
			m->target->a_time = m->target->orig_a_time;
	
	if(!m->has_mk && !m->code_time && !m->interface_time)
	{
		baltree_del(&this->modulexref,m->realname(),m->reallen());
		mem_delete m;
		if(toplev) this->top_mod = 0;
		return 0;				// Non existent module
	}
	this->modulelist.linkbefore(m);
	m->makevalid();
	return m;				// Valid module
}

void Parse::getcwd(){
	if( this->cwd.len() )
		return;
	dogetcwd( this->cwd );
}

parse_return Parse::getlist( const char *modname, int f ){
	String absolute;
	this->dellist();
	this->getcwd();
	absolute = modname;
	buildcompletename( absolute, this->cwd );
	
	if(f & MOD_ONE_COMPILE && f & MOD_ALL_COMPILE){
		fputs( this->Relative(absolute), stderr );
		fputs( ": Warning: parsing all modules\n", stderr );
		f &= ~MOD_ONE_COMPILE;
    }
	this->flags = f;
	if(this->ReadModule(absolute), !this->top_mod)
		return PARSE_NOMODULE;
    else if(!this->top_mod->isvalid()){
		this->dellist();
		return PARSE_FAIL;
    }
	this->startlist();
	return PARSE_OK;
}

String &Parse::Expand( String &line, int stripdollars ){
	size_t off1, off2 = 0;
	String env(10);
	while(off1 = line.chr('$',off2), off1 != BAD_SIZE_T){
		env.setlen(0);
		char m = line[off1+1];
		if(m != '(' && m != '{')
			if(line.len() == off1+1){ off2 = off1+1; continue; }
			else{
				if( line[ off1 + 1 ] == '$' ){
					off2 = off1 + 1;
					if( stripdollars ) line.del( off1, off2 );
					else off2++;
					continue;
				}
				env.set(0,line[off1+1]);
				line.del(off1,off1+2);
			}
		else if(off2 = line.chr(m == '(' ? ')' : '}',off1+2),
				off2 == BAD_SIZE_T)
			break;
		else{
			env.insert(0,line,off1+2,off2);
			line.del(off1,off2+1);
		}
		const String &val = this->getenv(env);
		line.insert(off1,val);
		off2 = off1 + val.len();
    }
	return line;
}

int Parse::CreateTargetInterface(module &m,action &act){
	String name(m.HalfMappedName());
	if( !name.len() )
		name = m.name();
	int f = 1;
	ExpandName( name, m.interface_namefmt );
	int fdout = open( name, O_RDWR|O_CREAT|O_TRUNC, 0666 );
	if(fdout == -1){
		act.type = ACT_ERROR;
		act.reset();
		act << "Cannot create " << this->Relative(name);
		return 0;
    }else{
		FILE *Out = fdopen( fdout, "w" );

		fputs( "/*\n"
			  "                               WARNING\n"
			  "             This file has been automatically created by mk\n"
			  "             If changes are made, they will be overwritten.\n"
			  "*/\n\n", Out );
		String inc;
		m.interface_needs.list.start();
		for(f = m.interface_needs.list.size(); f > 0; f--,
			m.interface_needs.list.fwd()){
			module *n = (module *)m.interface_needs.list.get();
			inc = n->name();
			ExpandName( inc, n->interface_namefmt );
			int fdin = open( inc, O_RDONLY );
			if(fdin == -1){
				act.type = ACT_ERROR;
				act.reset();
				act << "Cannot open " << this->Relative(inc);
				break;
			}else{
				FILE *In = fdopen( fdin, "r" );
				String line;
				fprintf( Out, "/* %s */\n", (const char *)inc );
				while( line.get( In ) ){
					fputs( line, Out );
					fputc( '\n', Out );
				}
				fclose( In );
			}
			close(fdin);
		}
		fflush( Out );
		if( ferror( Out ) ){
			act.type = ACT_ERROR;
			act.reset();
			act << "Failed writing " << this->Relative(name);
			f = 1;
		}
		fclose( Out );
    }
	close(fdout);
	return !f;
}

#define RSP_WANT_LIBS (1)
#define RSP_WANT_OBJECTS (2)

void Parse::CreateRspList(module &m,_balanced_tree *sorted,Dlist &list,
							   int WantLibs){
	module *new_m = &m;
	if((WantLibs & RSP_WANT_OBJECTS)
		&& !baltree_insert(sorted,&new_m,CPTRSZ,(void *)1)) return;
	m.direct_objects.list.start();
	for(int f = m.direct_objects.list.size(); f; f--,
		m.direct_objects.list.fwd()){
		new_m = (module *)m.direct_objects.list.get();
		if(new_m->target){
			if(new_m->target->namefmt && (WantLibs & RSP_WANT_LIBS)
			   && baltree_insert(sorted,&new_m,CPTRSZ,(void *)1)){
				this->CreateRspList(*new_m,sorted,list,RSP_WANT_LIBS);
				list.linkbefore(new_m);
			}
		}else this->CreateRspList(*new_m,sorted,list,WantLibs);
    }
    if(m.object_time && (WantLibs & RSP_WANT_OBJECTS)) list.linkbefore(&m);
}

void Parse::show_rsp_maps(action &act,Dlist &list){
	int f;
	for(f = list.size(), list.start(); f > 0; f--,list.fwd()){
		module *n = (module *)list.get();
		if(n->ismapped())
			act << "\n" << n->name() << " (from " << n->realname() << ")";
    }
}


int Parse::CreateRspFile( module &m, action &act, int toplev )
{
	String name(m.HalfMappedName());
	int HalfMapped = name.len(), ret = 0;
	Dlist list;
	_balanced_tree sorted;

	if( !HalfMapped )
		name = m.name();
	baltree_construct(&sorted,BALTREE_MALLOC_KEYS|BALTREE_FAST_KEYS);
	int flag = RSP_WANT_OBJECTS;
	if(!m.target->namefmt) flag |= RSP_WANT_LIBS;
	this->CreateRspList(m,&sorted,list,flag);
	
	if(this->map && !m.ismapped() && !HalfMapped && m.hasmapped()){
		if( toplev )
			fprintf( stderr, "Warning: Assuming %s as mapped\n", m.name() );
		else
		{
			act.type = ACT_ERROR;
			act.reset();
			act << m.name() << " cannot be loaded due to"
				<< " the following maps:";
			this->show_rsp_maps(act,list);
			return 0;
		}
	}
	
	ExpandName( name, this->defaults.value( RESPONSE_NAME ) );
	int fd = open( name, O_RDWR|O_CREAT|O_TRUNC, 0666 );
	if(fd == -1){
		act.type = ACT_ERROR;
		act.reset();
		act << "Cannot create " << this->Relative(name);
		baltree_destroy(&sorted);
		return ret;
	}else{
		FILE *Out = fdopen( fd, "w" );
		int haslines = 0, size;
		String s;
		const module *ThisModule;

		if(VERBOSE)
			fprintf( stderr, "%s contains the following:\n",
					(const char *)name );

		if( REVERSE_LOAD ) list.end();
		else list.start();
		size = list.size();

		while( size-- ){
			ThisModule = (module *)list.get();
			if(haslines){
				if(*m.responselinecont){
					fputs( m.responselinecont, Out );
					if(VERBOSE) fprintf( stderr, "%s", m.responselinecont );
				}
				fputc( '\n', Out );
				if(VERBOSE) fputc( '\n', stderr );
			}else haslines = 1;
			const char *Ext, *RealName = ThisModule->HalfMappedName();
			if(ThisModule->target && ThisModule != &m)
				Ext = ThisModule->target->namefmt;
			else{
				if( !ThisModule->objecthalfmapped() )
					RealName = ThisModule->name();
				Ext = ThisModule->object_namefmt;
			}
			if( !RealName ) RealName = ThisModule->name();
			this->SetTarget( RealName, Ext );
			this->Expand( s = m.responseline, 1 );
			fputs( s, Out );
			if( VERBOSE ) fputs( s, stderr );
			if( REVERSE_LOAD ) list.bkwd();
			else list.fwd();
		}
		baltree_destroy( &sorted );
		
		if( haslines ){
			fputc( '\n', Out );
			if(VERBOSE) fputc( '\n', stderr );
		}
		fflush( Out );
		if( VERBOSE ) fflush( stderr );
		if( ferror( Out ) ){
			act.type = ACT_ERROR;
			act.reset();
			act << "Failed writing " << this->Relative(name);
			ret = 1;
		}
		fclose( Out );
    }
	close(fd);
	return !ret;
}


void Parse::show_inc_maps(module &m,action &act){
	for(int first = 0; first < 2; first++){
		Dlist &h = first ? m.code_needs.list : m.interface_needs.list;
		h.start();
		for(int f = h.size(); f > 0; f--,h.fwd()){
			module *n = (module *)h.get();
			if(n->ismapped())
				act << "\n" << n->name() << " (from " << n->realname() << ")";
		}
    }
}

int Parse::CreateIncFile(module &m,action &act){
	String name(m.HalfMappedName());
	int f = 1;
	if( !name.len() )
		name = m.name();
	ExpandName( name, this->defaults.value( INCLUDE_NAME ) );
	int fd = open( name, O_RDWR|O_CREAT|O_TRUNC, 0666 );
	if(fd == -1){
		act.type = ACT_ERROR;
		act.reset();
		act << "Cannot create " << this->Relative(name);
		return 0;
    }else{
		FILE *Out = fdopen( fd, "w" );
		String s;
		int haslines = 0;
		char *cont = 0;
		if(VERBOSE)
			fprintf( stderr, "%s contains the following:\n",
					(const char *)name );
		m.interface_needs.list.start();
		for(f = m.interface_needs.list.size(); f > 0; f--,
			m.interface_needs.list.fwd()){
			module *n = (module *)m.interface_needs.list.get();
			this->SetTarget(n->name(),n->interface_namefmt);
			if(haslines){
				if(cont){
					fputs( cont, Out );
					if(VERBOSE) fprintf( stderr, "%s", cont );
				}
				if( VERBOSE ) fputc( '\n', stderr );
				fputc( '\n', Out );
			}else haslines = 1;
			s = n->includeline;
			fputs( this->Expand( s, 1 ), Out );
			if( VERBOSE ) fputs( s, stderr );
			cont = n->includelinecont;
		}
		if( m.interface_time ){
			this->SetTarget( m.name(), m.interface_namefmt );
			if(haslines){
				if(cont){
					fputs( cont, Out );
					if(VERBOSE) fprintf( stderr, "%s", cont );
				}
				fputc( '\n', Out );
				if( VERBOSE ) fputc( '\n', stderr );
			}else haslines = 1;
			fputs( this->Expand( s = m.includeline, 1 ), Out );
			if(VERBOSE) fputs( s, stderr );
			cont = m.includelinecont;
		}
		m.code_needs.list.start();
		for(f = m.code_needs.list.size(); f > 0; f--, m.code_needs.list.fwd()){
			module *n = (module *)m.code_needs.list.get();
			this->SetTarget( n->name(), n->interface_namefmt );
			if(haslines){
				if(cont){
					fputs( cont, Out );
					if(VERBOSE) fprintf( stderr, "%s", cont );
				}
				fputc( '\n', Out );
				if(VERBOSE) fputc( '\n', stderr );
			}else haslines = 1;
			fputs( this->Expand(s = n->includeline,1), Out );
			if(VERBOSE) fputs( s, stderr );
			cont = n->includelinecont;
		}
		if(haslines){
			fputc( '\n', Out );
			if(VERBOSE) fputc( '\n', stderr );
		}
		fflush( Out );
		if(VERBOSE) fflush( stderr );
		if( ferror( Out ) ){
			act.type = ACT_ERROR;
			act.reset();
			act << "Failed writing " << this->Relative(name);
			f = 1;
		}
		fclose( Out );
    }
	close(fd);
	return !f;
}

const char *Parse::nextmodule(){
	module *m = this->getmodule();
	return m ? m->name() : 0;
}

#define SRC (type & SOURCE_FILES)
#define TGT (type & TARGET_FILES)
#define TMP (type & TMP_FILES)


// Don't declare this inside ::nextfile - compilers such as WATCOM never
// delete it (stupid compiler).  This is probably because it doesn't actually
// instantiate it before main() - this is WRONG too.
static String nextfileName;

const char *Parse::nextfile(int type){
	static module *m = 0;
	static int file;
	do{
		if(m){
			nextfileName = m->name();
			for(file++; file < n_file_types; file++)
				switch((file_type)file){
					case C_F:
					if( m->derive && SRC ) continue;
					if(!m->code_time || !SRC) continue;
					ExpandName( nextfileName, m->code_namefmt );
					return (const char *)nextfileName;

					case D_F:
					if(!m->derive || !SRC) continue;
					ExpandName( nextfileName, m->derive->namefmt );
					return (const char *)nextfileName;

					case H_F:
					if( m->derive )
					{
						if( !TGT ) continue;
					}
					else if( m->target && m->target->namefmt )
					{
						if( !TGT || ( !RECURSING && m != this->top_mod ) )
							continue;
					}else if(!m->interface_time || !SRC) continue;
					ExpandName( nextfileName, m->interface_namefmt );
					return (const char *)nextfileName;

					case O_F:
					if(!m->object_time || !TGT) continue;
					ExpandName( nextfileName, m->object_namefmt );
					return (const char *)nextfileName;

					case A_F:
					if(!m->target || !TGT) continue;
					if(m != this->top_mod && !RECURSING) continue;
					if(m->target->namefmt)
						ExpandName( nextfileName, m->target->namefmt );
					else ExpandName( nextfileName,
									this->defaults.value( BINARY_NAME ) );
					return (const char *)nextfileName;

					case M_F:
					if(!m->has_mk || !SRC) continue;
					ExpandName( nextfileName, this->defaults.value( MK_NAME ) );
					return (const char *)nextfileName;

					case I_F:
					// Note: no code_time implies no object_time
					if( !m->code_time || !TMP ) continue;
					ExpandName( nextfileName,
								this->defaults.value( INCLUDE_NAME ) );
					return (const char *)nextfileName;

					case R_F:
					if(!m->target || !TMP) continue;
					if(m != this->top_mod && !RECURSING) continue;
					ExpandName( nextfileName,
								this->defaults.value( RESPONSE_NAME ) );
					return (const char *)nextfileName;
				}
		}
		file = -1;
    }while(m = this->getmodule(), m);
	return 0;
}

module *Parse::getmodule(){
	if(!this->top_mod) return 0;
	static int left;
	if(this->flags & MOD_STARTING){
		this->flags &= ~MOD_STARTING;
		this->modulelist.end();
		left = this->modulelist.size();
    }else this->modulelist.bkwd();
	if(!left--){
		this->startlist();
		return 0;
    }
	return (module *)this->modulelist.get();
}

enum a_decision { DO_EXE, DO_LIB, DO_COMPILE, DO_INT, DO_DERIVE, DO_UNKNOWN };

// Don't declare these inside ::getoutofdate - compilers such as WATCOM never
// delete them (stupid compiler).  This is probably because it doesn't actually
// instantiate it before main() - this is WRONG too.
static action OutOfDateAction;
static String MissingTargetFile;

action &Parse::getoutofdate( int &Priority, const char *external )
{
	static module *m;
	static int noload, nocompile;
	int toplev, allowed;
	a_decision decision;
	static a_decision next_decision = DO_UNKNOWN;
	
	if( !this->top_mod )
	{
		OutOfDateAction.type = ACT_ERROR;
		OutOfDateAction.reset();
		OutOfDateAction << "No list has been built";
		return OutOfDateAction;
	}
	
	if(this->flags & MOD_STARTING)
	{
		noload = nocompile = 0;
		MissingTargetFile.setlen(0);
		next_decision = DO_UNKNOWN;
	}
	
	while( ( m = next_decision == DO_UNKNOWN ? this->getmodule() : m ), m )
	{
		toplev = m == this->top_mod;

		switch( decision = next_decision, decision )
		{
			case DO_COMPILE:
				next_decision = m->target && m->target->a_time == OUTOFDATE ?
					DO_EXE : DO_UNKNOWN;
				break;

			case DO_EXE:
			case DO_LIB:
				next_decision = DO_UNKNOWN;
				break;

			default:
				if( m->target && m->target->namefmt )
					if( m->interface_time == OUTOFDATE )
					{
						decision = DO_INT;
						if( m->target->a_time == OUTOFDATE )
							next_decision = DO_LIB;
						else
							next_decision = DO_UNKNOWN;
					}
					else if( m->target->a_time == OUTOFDATE )
					{
						decision = DO_LIB;
						next_decision = DO_UNKNOWN;
					}
					else
					{
						next_decision = DO_UNKNOWN;
						if( VERBOSE )
							fprintf( stderr, "%s: Up to date\n", m->name() );
						continue;
					}
				else if( m->code_time == OUTOFDATE )
				{
					decision = DO_DERIVE;
					next_decision = DO_COMPILE;
				}
				else if( m->object_time == OUTOFDATE )
				{
					decision = DO_COMPILE;
					next_decision = m->target ? DO_EXE : DO_UNKNOWN;
				}
				else if( m->target && m->target->a_time == OUTOFDATE )
				{
					decision = DO_EXE;
					next_decision = DO_UNKNOWN;
				}
				else
				{
					next_decision = DO_UNKNOWN;
					if( VERBOSE )
						fprintf( stderr, "%s: Up to date\n", m->name() );
					continue;
				}
				break;
		}

		// We know that "decision" *should* be done, but can we do it ?
		switch( decision )
		{
			case DO_EXE:
			case DO_LIB:
				allowed = LOADING && ( toplev || RECURSING );
				break;

			case DO_COMPILE:
			case DO_INT:
			case DO_DERIVE:
				allowed = COMPILING_ALL || ( COMPILING_ONE && ( toplev ||
					( RECURSING && m->target && m->target->namefmt ) ) );
				break;

			default:
				// Impossible
				continue;
		}

		if( !allowed )
		{
			if( m->target && m->target->namefmt )
			{
				if( REMOVING ) continue;

				// Check for a showstopper because we have no recursion flag
				if( decision == DO_LIB )
				{
					if( !m->target->orig_a_time && !MissingTargetFile.len() )
					{
						// Moan about missing LIB later
						MissingTargetFile = m->name();
						ExpandName( MissingTargetFile, m->target->namefmt );
						noload = 1;
					}
				}
				else if(!m->orig_interface_time && !MissingTargetFile.len())
				{
					// Moan about missing INTERFACE later
					MissingTargetFile = m->name();
					ExpandName( MissingTargetFile, m->interface_namefmt );
					nocompile = 1;
				}
			}
			else if( decision == DO_COMPILE && !m->object_time
					&& !MissingTargetFile.len() )
			{
				// Moan about missing LIB later
				MissingTargetFile = m->name();
				ExpandName( MissingTargetFile, m->object_namefmt );
				noload = 1;
			}
			// Note, we don't care about DO_DERIVE as the DO_COMPILE will
			// fail for the same reasons and will block the load anyway.
			// We don't care about DO_EXE as they aren't gonna affect
			// anything above them

			if( VERBOSE )
			{
				String Name;
				switch( decision )
				{
					case DO_EXE:		Name = "executable"; break;
					case DO_LIB:		Name = "library"; break;
					case DO_COMPILE:	Name = "object"; break;
					case DO_INT:		Name = "library header"; break;
					case DO_DERIVE:		Name = "source and header"; break;
					default:			Name = "*** FAULT ***"; break;
				}
				fprintf( stderr, "Cannot build %s for %s due to run flags\n",
						(const char *)Name, m->name() );
			}
			continue;
		}
		
		char *namefmt = 0;
		time_t last_time, target_time;
		
		Priority = 4;
		switch(decision)
		{
			case DO_DERIVE:
				if( VERBOSE ) fputs( "Decision is DERIVE\n", stderr );
				namefmt = m->code_namefmt;
				last_time = m->orig_code_time;
				target_time = m->code_time;
				Priority = 0;
				// for OutOfDateAction.Compiled()
				OutOfDateAction.type = ACT_DERIVE;
				break;

			case DO_INT:
				if( VERBOSE ) fputs( "Decision is INTERNAL\n", stderr );
				namefmt = m->interface_namefmt;
				last_time = m->orig_interface_time;
				target_time = m->interface_time;
				Priority = 1;
				break;

			case DO_COMPILE:
				if( VERBOSE ) fputs( "Decision is COMPILE\n", stderr );
				namefmt = m->object_namefmt;
				last_time = m->orig_object_time;
				target_time = m->object_time;
				Priority = 2;
				// for OutOfDateAction.Compiled()
				OutOfDateAction.type = ACT_COMPILE;
				break;

			case DO_LIB:
				if( VERBOSE ) fputs( "Decision is LIBRARY\n", stderr );
				namefmt = m->target->namefmt;
				Priority = 3;
				// yes, continue to next case

			case DO_EXE:
				if( !namefmt )
				{
					if( VERBOSE ) fputs( "Decision is EXEC\n", stderr );
					namefmt = this->defaults.value( BINARY_NAME );
				}
				last_time = m->target->orig_a_time;
				target_time = m->target->a_time;
				// for OutOfDateAction.Compiled()
				OutOfDateAction.type = ACT_LOAD;
				break;

			default:
				if( VERBOSE ) fputs( "Decision is *** FAULT ***\n", stderr );
				last_time = target_time = 0;	// Never reached
		}
		
		OutOfDateAction.Setup( m, last_time, namefmt, QUIET );

		if( !external && !REMOVING )
			OutOfDateAction.Lock( this->defaults.value( LOCK_NAME ) );
		
		if( OutOfDateAction.Compiled( VERBOSE ) )
		{
			if( OutOfDateAction.IsLockedByMe() )
				OutOfDateAction.UnLock();

			if( OutOfDateAction.IsLocked( this->defaults.value( LOCK_NAME ) ) )
			{
				// Someone else has created the target but not yet removed the
				// lock file - some compilers do this all the time by creating
				// the target as a zero length file, then compiling, then
				// writing the results to the file
				OutOfDateAction << this->Relative( m->name() )
					<< ": locked by " << OutOfDateAction.LockedBy();
			}
			else if(REMOVING)
			{
				String Name( m->name() );
				ExpandName( Name, namefmt );
				OutOfDateAction << Name << ": Just built - not removed";
			}
			else
				// After all that, someone else has done it !
				continue;
			OutOfDateAction.type = ACT_WARNING;
			return OutOfDateAction;
		}

		// So get to work !
		
		if( REMOVING )
		{
			if( !last_time ) continue;
			String rmfile( m->name() );
			ExpandName( rmfile, namefmt );
			if( external )
				OutOfDateAction << external << " " << (char *)rmfile;
			else
			{
				if( unlink( rmfile ) )
					perror( rmfile );
				OutOfDateAction << "Remove " << this->Relative( rmfile );
			}
			OutOfDateAction.type = ACT_INTERNAL;
			return OutOfDateAction;
		}

		if( ( ( decision == DO_LIB || decision == DO_EXE ) && noload )
			|| ( decision == DO_COMPILE && nocompile ) )
		{
			String Name( m->name() );
			if( OutOfDateAction.IsLockedByMe() )
				OutOfDateAction.UnLock();
			ExpandName( Name, namefmt );
			OutOfDateAction.type = ACT_ERROR;
			OutOfDateAction.reset();
			OutOfDateAction << this->Relative( Name ) << ": component target ";
			OutOfDateAction << this->Relative( MissingTargetFile )
				<< " missing";
			return OutOfDateAction;
		}
		
		switch(decision)
		{
			case DO_EXE:
			case DO_LIB:
				if( OutOfDateAction.IsLockedByMe()
					&& !this->CreateRspFile(*m,OutOfDateAction, toplev ) )
				{
					OutOfDateAction.UnLock();
					return OutOfDateAction;
				}
				this->SetTarget( m->name(), namefmt,
								this->defaults.value( RESPONSE_NAME ) );
				OutOfDateAction << m->load;
				this->Expand( OutOfDateAction.exec_command, 1 );
				OutOfDateAction.type = ACT_LOAD;
				return OutOfDateAction;
			
			case DO_INT:
			{
				String Name( m->name() );
				ExpandName( Name, namefmt );
				OutOfDateAction.type = ACT_INTERNAL;
				if( external )
					OutOfDateAction << external << " " << Name;
				else if( OutOfDateAction.IsLockedByMe() )
				{
					if( OutOfDateAction.Compiled( VERBOSE ) )
					{
						OutOfDateAction.UnLock();
						continue;
					}
					else if( !this->CreateTargetInterface(*m,OutOfDateAction) )
					{
						OutOfDateAction.type = ACT_ERROR;
						OutOfDateAction.reset();
						OutOfDateAction << "\n" << this->Relative( Name )
							<< " missing";
						OutOfDateAction.UnLock();
					}
					else
						OutOfDateAction << "Build " << Name;
				}
				else
				{
					OutOfDateAction << this->Relative( m->name() )
						<< ": locked by " << OutOfDateAction.LockedBy();
					OutOfDateAction.type = ACT_WARNING;
				}
				return OutOfDateAction;
			}
			
			case DO_COMPILE:
				// Am I including code that is mapped in a module that isn't ?
				if( this->map && !m->ismapped() && m->hasmapped()
					&& !m->HalfMappedName() && m->objecthalfmapped() )
				{
					if( toplev )
						fprintf( stderr, "Warning: Assuming %s as mapped\n",
								m->name() );
					else
					{
						// This can only happen if there's nowhere to half-map
						// the module to.
						OutOfDateAction.type = ACT_ERROR;
						OutOfDateAction.reset();
						OutOfDateAction << m->name()
							<< " cannot be compiled due to the following maps:";
						this->show_inc_maps(*m,OutOfDateAction);
						OutOfDateAction.UnLock();
						return OutOfDateAction;
					}
				}
				if( OutOfDateAction.IsLockedByMe()
					&& !this->CreateIncFile(*m,OutOfDateAction))
				{
					OutOfDateAction.UnLock();
					return OutOfDateAction;
				}
				this->SetTarget( m->name(), namefmt,
								this->defaults.value( INCLUDE_NAME ),
								m->code_namefmt );
				OutOfDateAction << m->compile;
				this->Expand( OutOfDateAction.exec_command, 1 );
				OutOfDateAction.type = ACT_COMPILE;
				return OutOfDateAction;

			case DO_DERIVE:
				this->SetTarget( m->name(), namefmt,	// $@
								m->interface_namefmt,	// $&
								m->derive->namefmt );	// $<
				OutOfDateAction << m->derive->compile;
				this->Expand( OutOfDateAction.exec_command, 1 );
				OutOfDateAction.type = ACT_DERIVE;
				return OutOfDateAction;

			default:
				OutOfDateAction.UnLock();
				OutOfDateAction.type = ACT_ERROR;
				OutOfDateAction << "ERROR: What the hell happened there?\n";
				return OutOfDateAction;
		}
    }

	OutOfDateAction.type = ACT_DONE;
	OutOfDateAction.reset();
	return OutOfDateAction;
}

enum command_type { DIRECTIVES, VALUES, ASSIGNMENTS };
enum assign_type { IGNORE, DEFINE, SET, SETENV };

int Parse::startup(String &rcfile,int &flags){
	int fd = open( rcfile, O_RDONLY );
	int err = errno, lineno = 0;
	size_t offset = 0;
	char ch;
	default_type t = INVALID_TYPE;
	assign_type assignment = IGNORE;
	command_type type = ASSIGNMENTS;
	
	this->GiveUp = 15;
	this->getcwd();
	buildcompletename( rcfile, this->cwd );
	if(fd == -1){
		fputs( "Cannot open startup file ", stderr );
		errno = err;
		perror(this->Relative(rcfile));
		return 0;
    }else{
		FILE *In = fdopen( fd, "r" );
		String line;
		int dirlen = 0;
		
		while(lineno++, line.get( In ))
			if(line[0] == ';') continue;
			else if(line[0] == '#'){
				line.del(0,1);
				line.unpad();
				if(!strcmp(line,"directives")) type = DIRECTIVES;
				else if(!strcmp(line,"values")) type = VALUES;
				else if(!strcmp(line,"assign")) type = ASSIGNMENTS;
				else if(!line.ncmp("include",7) &&
						(line.del(0,7), ch = line[0],
						 ch == ' ' || ch == '\t')){
					line.unpad();
					dirlen = buildcompletename( line, rcfile, dirlen );
					if(!this->startup(line,flags)){
						fclose( In );
						close( fd );
						return 0;
					}
				}
				else
					fprintf( stderr, "%s: Invalid command: %s\n",
							(const char *)this->Relative(rcfile),
							(const char *)line );
			}else{
				line.unpad();
				if(!line.len()) continue;
				offset = line.chr('=');
				if(offset == BAD_SIZE_T){
					fprintf( stderr, "%s: %d: %s: Invalid line\n",
							(const char *)this->Relative(rcfile), lineno,
							(const char *)line );
					continue;
				}
				String command(line,offset);
				line.del(0,offset+1);
				command.unpad();
				line.unpad();
				if(type == ASSIGNMENTS)
					if(!strcmp(command,"IGNORE")) assignment = IGNORE;
					else if(!strcmp(command,"ECHO")){
						fputs( line, stderr );
						fputc( '\n', stderr );
						continue;
					}else if(!command.ncmp("DEFINE",6)){
						offset = 5;
						while(ch = command[++offset], ch == ' ' || ch == '\t')
							;
						command.del(0,offset);
						if(offset == 6 || !command.len()){
							fprintf( stderr, "%s: %d: %s: Invalid "
									"DEFINE command\n",
									(const char *)this->Relative(rcfile),
									lineno, (const char *)command );
							continue;
						}
						offset = 0;
						while(ch = command[offset], ch != ' ' && ch != '\t'
							  && offset < command.len())
							offset++;
						String def_t(command,offset);
						if(t = this->defaults.type(def_t), t == INVALID_TYPE){
							fprintf( stderr, "%s: %d: %s: Invalid "
									"DEFINE type\n",
									(const char *)this->Relative(rcfile),
									lineno, (const char *)def_t );
							continue;
						}
						while(ch = command[++offset], ch == ' ' || ch == '\t')
							;
						command.del(0,offset);
						assignment = DEFINE;
					}
					else if( !command.ncmp( "SET", 3 ) )
					{
						if( !command.ncmp( "SETENV", 6 ) )
						{
							assignment = SETENV;
							offset = 5;
						}
						else
						{
							assignment = SET;
							offset = 2;
						}
						while(ch = command[++offset], ch == ' ' || ch == '\t')
							;
						command.del(0,offset);
						if(offset == 3 || !command.len()){
							fprintf( stderr, "%s: SET%s command: %d: %s: Invalid\n",
									(const char *)this->Relative(rcfile),
									assignment == SET ? "" : "ENV",
									lineno, (const char *)command );
							continue;
						}
					}else{
						fprintf( stderr, "%s: %d: %s: Invalid command\n",
								(const char *)this->Relative(rcfile), lineno,
								(const char *)command );
						continue;
					}
				else if(t = this->defaults.type(command), t == INVALID_TYPE){
					fprintf( stderr, "%s: %d: %s: Invalid default ",
							(const char *)this->Relative(rcfile), lineno,
							(const char *)command );
					if(type == DIRECTIVES) fputs( "directive\n", stderr );
					else fputs( "value\n", stderr );
					continue;
				}
				if(type == DIRECTIVES && line[0] == '#'){
					line.del(0,1);
					line.unpad();
				}
				this->dontexpandtarget();
				this->Expand(line);
				if(type == VALUES) this->defaults.set_value(t,line);
				else if(type == DIRECTIVES)
					this->defaults.set_directive(t,line);
				else if(assignment == DEFINE)
					this->defaults.define(t,command,line,
										  this->Relative(rcfile),lineno);
				else if(assignment == IGNORE){
					const _balanced_tree_key *k;
					baltree_setafter(&this->ignore,line,line.len()+1);
					if(baltree_prev(&this->ignore)){
						k = baltree_key(&this->ignore);
						if(line.len() >= k->length
						   && !memcmp(k->data,line,k->length))
							continue;
					}
					if(baltree_next(&this->ignore)){
						k = baltree_key(&this->ignore);
						if(line.len() <= k->length
						   && !memcmp(k->data,(const char *)line,line.len()))
							baltree_del(&this->ignore,k->data,k->length);
					}
					baltree_insert(&this->ignore,(const char *)line,
								   line.len(),(void *)1);
				}else if(assignment == SET){
					// SET 'command' = 'line'
					if(!strcmp(command,"MAPPING")){
						if(!this->set_map(line,flags & MOD_VERBOSE)){
							fclose( In );
							close( fd );
							return 0;
						}
					}else if( !strcmp( command, "LOAD" ) ){
						if( line == "FORWARD" ){
							if( flags & MOD_REVERSE_LOAD )
								fputs( "Warning: REVERSE LOAD canceled in "
									  "startup file\n", stderr );
							flags &= ~MOD_REVERSE_LOAD;
						}else if( line == "BACKWARD" )
							flags |= MOD_REVERSE_LOAD;
						else{
							fputs( "Only values of FORWARD and BACKWARD may"
								  " be used for SET LOAD\n", stderr );
							fclose( In );
							close( fd );
							return 0;
						}
					}else if( !strcmp( command, "GIVEUP" ) ){
						for( size_t f = 0; f < line.len(); f++ )
							if( line[f] < '0' || line[f] > '9' )
							{
								fputs( "SET GIVEUP: Value must be a "
										"positive numeric integer\n",
										stderr );
								fclose( In );
								close( fd );
								return 0;
							}
						this->GiveUp = atoi( line );
					}else{
						fprintf( stderr, "SET %s: Unknown assignment\n",
								(char *)command );
						fclose( In );
						close( fd );
						return 0;
					}
				}else if(assignment == SETENV)
					// SETENV 'command' = 'line'
					this->putenv( command, line );
			}
		fclose( In );
	}
	close(fd);
	return 1;
}

int Parse::mapstring(const String &line,const String &dir,int verbose,
					 const char *file,int lineno){
	String from(line), to;
	from.unpad();
	const char *ch = from;
	while(*ch != ' ' && *ch != '\t' && *ch) ch++;
	if(!*ch){
		if(lineno)
			fprintf( stderr, "%s: %d: Invalid line\n", (const char *)file,
					lineno );
		return 0;
	}
	to = ch + 1;
	to.unpad();
	this->Expand(to,1);
	buildcompletename( to, dir );
	from.setlen(ch - (const char *)from);
	this->Expand(from,1);
	buildcompletename( from, dir );
	if(from == to){
		if(lineno) fprintf( stderr, "%s: %d: ", (const char *)file, lineno );
		fputs( "Cannot map ", stderr );
		fputs( from, stderr );
		fputs( " onto itself\n", stderr );
		return 1;
	}
	if(verbose)
		fprintf( stderr, "Map prefix %s to %s\n", (const char *)from,
				(const char *)to );
	if(!this->map) this->map = mem_new mapping;
	this->map->add(from,to);
	return 1;
}

int Parse::set_map(const String &omapfile,int verbose){
	String mapfile(omapfile);
	this->getcwd();
	buildcompletename( mapfile, this->cwd );
	this->dontexpandtarget();
	int fd = open( mapfile, O_RDONLY );
	if(fd == -1){
		int olderrno = errno;
		if(!this->mapstring(omapfile,this->cwd,verbose)){
			errno = olderrno;
			perror(mapfile);
			return 0;
		}
		return 1;
	}else{
		FILE *In = fdopen( fd, "r" );
		int lineno = 0;
		String dir(mapfile), from;
		dir.setlen(dir.rchr(SLASH));
		while(lineno++, from.get( In )){
			if(from[0] == ';') continue;
			if(!this->mapstring(from,dir,verbose,mapfile,lineno)){
				fclose( In );
				close( fd );
				return 0;
			}
		}
		fclose( In );
	}
	close(fd);
	return 1;
}

void Parse::rm_map(){
	if(this->map){ mem_delete this->map; this->map = 0; }
}

#if defined( MSDOS ) | defined( OS2 )
char drivecmd[3] = " :";
#endif

void joberror(const char *prog,int err){
	if(err & 255)
		switch( err ){
			case MODULE_NOT_COMPILED:
			fputs( prog, stderr );
			fputs( " failed to produce target\n", stderr );
			break;

			case JOB_FAILED_WAIT:
			// The message has already come out (jobs::wait)
			break;

			case JOB_FAILED_UNLOCK:
			// The message has already come out (action::unlock)
			break;

			default:
			fprintf( stderr, "%s died due to signal %d\n",
					(const char *)prog, err & 255 );
		}
    else
		fprintf( stderr, "%s returned %d\n", (const char *)prog, err >> 8 );
}

#ifndef MSDOS
jobs *globjob;
const action *globact;
static Sig_Handler_Return dokill(int sig){
	fprintf( stderr, "Received signal %d\n", sig );
	if(globjob) globjob->killall(sig);
	globjob = 0;
}
#endif

int ResetRetries( int &SleepTime, jobs& job, action &act )
{
	if( SleepTime ){
		for( int f = 0; f < SleepTime; f++ ){
			sleep(1);
#ifndef MSDOS
			if( f > 20 || !globjob )
#else
			if( f > 20 )
#endif
				break;
		}
		SleepTime <<= 1;
	}else SleepTime = 1;
	int ret = job.finish( act );
	if( ret )
		joberror( act.prog(), ret );

	return ret;
}
	
int Parse::compile(const char *external,int parallel){
	int ret = 0, changeddir = 0;
	int Priority, RetryPriority = -1, RetrySleep = 0, GiveUpValue = 0;
	String lastcd(this->cwd);
#if defined( MSDOS ) | defined( OS2 )
	unsigned lastdrive;
	DOSgetdisk(&lastdrive);
	unsigned origdrive = lastdrive;
#endif
	jobs job(parallel);
	action finished;

#ifndef MSDOS
	signals sig(SIGHUP,SIGINT,SIGQUIT,SIGTERM,0);
	globjob = &job;
	sig.stack(dokill);
#endif
		
	while(1)
	{
#ifndef MSDOS
		if( !globjob ) break;
#endif
		action &act = this->getoutofdate( Priority, external );

		if(act.type == ACT_DONE){ ret = 0; break; }
		if(act.type == ACT_LOAD || act.type == ACT_COMPILE
			|| act.type == ACT_DERIVE )
		{
			if( VERBOSE )
			{
				fputs( act.name(), stderr );
				fputs( " is to be built\n", stderr );
			}
			if( RetryPriority != -1 &&
			   ( !Priority || Priority > RetryPriority ) )
			{
				if( act.IsLockedByMe() ) act.UnLock();
				if( this->GiveUp && ++GiveUpValue >= this->GiveUp )
				{
					fprintf( stderr, "Giving up on %s\n",
							(char *)this->Relative( act.name() ) );
					break;
				}
				if( ResetRetries( RetrySleep, job, act ) )
					break;
				RetryPriority = -1;
				this->flags |= MOD_STARTING;
				continue;
			}
			else if( !act.IsLockedByMe() && !external )
			{
				// if it's my processes lock, it's ok !
				if( act.type == ACT_LOAD && job.isrunning(act) )
				{
					int ret;
					if( ret = job.waitfor(act), ret )
					{
						joberror(act.prog(),ret);
						break;
					}
					if( ResetRetries( RetrySleep, job, act ) )
						break;
					RetryPriority = -1;
					this->flags |= MOD_STARTING;
					GiveUpValue = 0;
					// Something finished. Go back and see if everything's done
				}
				else
				{
					fprintf( stderr, "%s: Locked by %s\n",
							(char *)this->Relative(act.name()),
							(char *)act.LockedBy() );
					if( !Priority )
					{
						RetryPriority = -1;
						this->flags |= MOD_STARTING;
						if( ResetRetries( RetrySleep, job, act ) )
							break;
					}
					else
						RetryPriority = Priority;
					// Keep looking for something to do
				}
				continue;
			}
			else
			{
				RetrySleep = 0;
				// We're doing something so definitely don't give up
				GiveUpValue = -1;
			}
			
			if(lastcd != act.dir())
			{
				changeddir = this->cwd != act.dir();
#if defined( MSDOS ) | defined( OS2 )
				if(act.dir()[0] - 'a' != lastdrive)
				{
					lastdrive = act.dir()[0] - 'a';
					if(!QUIET)
					{
						*drivecmd = act.dir()[0];
						prcommand(drivecmd);
					}
					if(!external) DOSsetdisk(lastdrive);
				}
				if(!QUIET) pr2command("cd ",(const char *)act.dir() + 2);
#else
				if(!QUIET) pr2command("cd ",act.dir());
#endif
				lastcd = act.dir();
				if(!external) chdir(lastcd);
			}
			
			const char *namefmt = 0;
			if(!KEEP && act.type != ACT_DERIVE )
				namefmt = act.type == ACT_LOAD ?
					this->defaults.value( RESPONSE_NAME ) :
					this->defaults.value( INCLUDE_NAME );
			
			if(ret = job.execute(finished,act,namefmt,Priority,QUIET,
								 external ? 0 : 1), ret)
			{
				joberror(finished.prog(),ret);
				break;
			}
		}
		else if(act.type == ACT_ERROR)
		{
			fputs( act.command(), stderr );
			fputc( '\n', stderr );
			ret = 1;		// Kills all outstanding jobs
			break;
		}
		else if( !QUIET && *act.command() )
		{
			fputs( act.command(), stderr );
			fputc( '\n', stderr );
			// At the moment, all ACT_WARNINGs are "locked by" messages.
			// We'd better go back and have a look, otherwise either the
			// comile will fail or we'll try to link the object before
			// it's built - both bad news !
			RetryPriority = Priority;
		}
		if( act.IsLockedByMe() ) act.UnLock();
	}
	
#ifndef MSDOS
	if( globjob )
	{
#endif
		if(!ret && (ret = job.finish(finished), ret))
			joberror(finished.prog(),ret);
#ifndef MSDOS
		if(ret) job.killall(SIGTERM);
	}
	sig.restore();
#endif
	
	if(changeddir){
#if defined( MSDOS ) | defined( OS2 )
		if(origdrive != lastdrive){
			if(!QUIET){
				*drivecmd = this->cwd[0];
				prcommand(drivecmd);
			}
			if(!external) DOSsetdisk(this->cwd[0] - 'a');
		}
		if(!QUIET) pr2command("cd ",(const char *)this->cwd + 2);
#else
		if(!QUIET) pr2command("cd ",this->cwd);
#endif
		if(!external) chdir(this->cwd);
	}
	return !ret;
}
