#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <string.h>
#include <stdio.h>

static char Buffer[ 1024 ];
const char *VerboseString = "tcpchat version 1.3, freeware, written by <brian@awfulhak.org>\n";

static int Usage()
{
    fprintf(stderr, "Usage: tcpchat [ -sv[v] ] [ -t Timeout ] [ -r [n[.m]] ]\n");
    fprintf(stderr, "               Host Port [ Expect [ Send ] ]...\n");
    fprintf(stderr, "       where   -s makes tcpchat completely silent\n");
    fprintf(stderr, "               -v displays received and sent data\n");
    fprintf(stderr, "               -vv displays program copyright info\n");
    fprintf(stderr, "               -t Timeout specifies the timeout in seconds (default 5)\n");
    fprintf(stderr, "               -r n.m defines repetition characteristics\n");
    fprintf(stderr, "                  n specifies how many times to repeat (0 is infinite)\n");
    fprintf(stderr, "                  m specifies the argument to repeat from\n");
    fprintf(stderr, "               The default is -r 1.0\n");
    return 1;
}

int Send(int fd, const char *Data, int Verbose)
{
    int Len = strlen(Data);
    strcpy(Buffer, Data);
    Buffer[Len] = '\n';
    if (Verbose)
        write(1, Buffer, Len + 1);
    if (write(fd, Buffer, Len + 1) == -1) {
        perror("write");
        return 0;
    }
    return 1;
}

static int TimedOut = 0;
int Timeout(int Sig)
{
    TimedOut = 1;
}

int Expect(int fd, const char *Data, unsigned TimeoutVal, int Verbose)
{
    int Result;

#ifdef ARCHAIC
    TimedOut = 0;
    if (TimeoutVal) {
        signal(SIGALRM, Timeout);
        alarm(TimeoutVal);
    }
#else
    struct sigaction act, oact;

    TimedOut = 0;
    if (TimeoutVal) {
        act.sa_handler = (sig_t)Timeout;
        sigemptyset(&act.sa_mask);
        act.sa_flags = 0;
        sigaction(SIGALRM, &act, &oact);
        alarm(TimeoutVal);
    }
#endif

    while (Result = read(fd, Buffer, sizeof(Buffer)), Result != -1) {
        if (Verbose)
                write(1, Buffer, Result);
        Buffer[Result] = '\0';
        if (TimedOut)
            break;
        else if (strstr(Buffer, Data)) {
            Result = 1;
            break;
        }
    }

    if (TimedOut)
        Result = 0;

    if (TimeoutVal)
#ifdef ARCHAIC
        signal (SIGALRM, SIG_DFL);
#else
        sigaction(SIGALRM, &oact, 0);
#endif
    return Result;
}

main(int argc, char **argv)
{
    struct servent *s = 0;
    struct hostent *h;
    struct sockaddr_in sock;
    int f, fd, ExpectArg, Verbose = 0, Silent = 0;
    unsigned nRepeat = 1, RepeatFrom = 0;
    unsigned TimeoutVal = 5;
    char *Ptr, *DoneWord = "x", *Arg;
#ifndef ARCHAIC
    struct sigaction act, oact;
#endif

    for (f = 1; f < argc; f++)
        if (*argv[f] == '-') {
            for (Ptr = argv[f] + 1; *Ptr; Ptr++)
                switch (*Ptr) {
                    case 's':
                        Silent = 1;
                        Verbose = 0;
                        break;

                    case 'v':
                        if (Verbose)
                            fputs(VerboseString, stderr);
                        Silent = 0;
                        Verbose = 1;
                        break;
    
                    case 'r':
                        Arg = Ptr[1] ? Ptr + 1 : argv[++f];
                        nRepeat = (unsigned)atoi(Arg);
                        Arg = index(Arg, '.');
                        if (Arg)
                            RepeatFrom = (unsigned)atoi(Arg+1);
                        Ptr = DoneWord;
                        break;

                    case 't':
                        TimeoutVal = (unsigned)atoi
                            (Ptr[1] ? Ptr + 1 : argv[++f]);
                        Ptr = DoneWord;
                        break;
    
                    default:
                        return Usage();
                }
        }
        else
            break;


    if (argc < f + 2)
        return Verbose ? 0 : Usage();

    h = gethostbyname(argv[f]);
    if (!h)
        h = gethostbyaddr(argv[f], strlen( argv[f] ), AF_INET);
    if (!h || h->h_addrtype != AF_INET) {
        fprintf(stderr, "%s isn't an internet address!\n", argv[f]);
        return Usage();
    }

    f++;
    if (strspn(argv[f], "0123456789") == strlen(argv[f]))
        sock.sin_port = htons(atoi(argv[f]));
    else if (s = getservbyname(argv[f], "tcp"), !s) {
        fprintf(stderr, "%s isn't a valid socket!\n", argv[f]);
        return Usage();
    }
    else
        sock.sin_port = s->s_port;

#ifndef ARCHAIC
    sock.sin_len = sizeof(sock);
#endif
    sock.sin_family = AF_INET;
    sock.sin_addr.s_addr = *(u_long *)h->h_addr_list[0];

    if (fd = socket(AF_INET, SOCK_STREAM, 0), fd < 0) {
        fprintf(stderr, "Cannot create internet socket\n");
        return 2;
    }

    TimedOut = 0;
    if (TimeoutVal) {
#ifdef ARCHAIC
        signal(SIGALRM, Timeout);
        alarm(TimeoutVal);
#else
        act.sa_handler = (sig_t)Timeout;
        sigemptyset(&act.sa_mask);
        act.sa_flags = 0;
        sigaction(SIGALRM, &act, &oact);
        alarm(TimeoutVal);
#endif
    }

    if (connect(fd, (struct sockaddr *)&sock, sizeof(sock)) < 0) {
        if (TimeoutVal)
#ifdef ARCHAIC
            signal(SIGALRM, SIG_DFL);
#else
            sigaction(SIGALRM, &oact, 0);
#endif
        if (!Silent) {
            if (TimedOut)
                fputs("Timeout: ", stderr);
            fprintf(stderr, "Cannot connect to %s:%s\n", argv[f-1], argv[f]);
        }
        close( fd );
        return 3;
    }

    ExpectArg = ++f % 2;
    RepeatFrom += f;
    do {
        for (; f < argc; f++)
            if (f % 2 == ExpectArg) {
                if (*argv[f] && !Expect(fd, argv[f], TimeoutVal, Verbose)) {
                    if (!Silent)
                        fprintf(stderr,
                            "Timeout(%d) waiting for \"%s\" from %s:%d\n",
                            TimeoutVal, argv[f], h->h_name, sock.sin_port);
                    close(fd);
                    return 4;
                }
            } else if (!Send(fd, argv[f], Verbose)) {
                fprintf(stderr, "Failed to send \"%s\"\n", argv[f]);
                close(fd);
                return 5;
            }
        f = RepeatFrom;
    } while (!nRepeat || --nRepeat);

    close(fd);
    
    return 0;
}
